# Ready Player Me API Client - Документация

Полнофункциональный Swift клиент для работы с Ready Player Me REST API.

## 📋 Содержание

- [Аутентификация и создание пользователей](#аутентификация-и-создание-пользователей)
- [Работа с аватарами](#работа-с-аватарами)
- [Работа с пользователями](#работа-с-пользователями)
- [Работа с ассетами](#работа-с-ассетами)
- [Параметры рендеринга](#параметры-рендеринга)

---

## 🔐 Аутентификация и создание пользователей

### Создание анонимного пользователя

```swift
let api = ReadyPlayerMeAPI()

let response = try await api.createAnonymousUser(appId: nil)
print("User ID: \(response.id)")
print("Token: \(response.token)")

// Используйте токен в avatar creator URL
```

### Создание пользователя с email

```swift
let response = try await api.createUser(
    email: "user@example.com",
    name: "John Doe",
    appId: nil
)
print("User ID: \(response.id)")
print("Token: \(response.token)")
```

### Email аутентификация (2-step flow)

**Шаг 1: Запрос кода для входа**

```swift
let response = try await api.requestLoginCode(
    email: "user@example.com",
    appId: nil
)
print("Message: \(response.message)")
// Пользователь получит код на email
```

**Шаг 2: Вход с кодом**

```swift
let response = try await api.login(
    email: "user@example.com",
    code: "123456"
)

// Сохраните токены
let accessToken = response.token
let refreshToken = response.refreshToken
let userInfo = response.user

// Сохранить для последующего использования
UserDefaults.standard.set(accessToken, forKey: "rpm_access_token")
UserDefaults.standard.set(refreshToken, forKey: "rpm_refresh_token")
```

### Обновление токена

```swift
let response = try await api.refreshToken(refreshToken: savedRefreshToken)

// Обновите сохраненные токены
let newAccessToken = response.token
let newRefreshToken = response.refreshToken
```

### Обновление информации пользователя

```swift
let updatedUser = try await api.updateUser(
    userId: "user-id",
    name: "New Name",
    email: "newemail@example.com"
)
```

### Удаление пользователя

```swift
try await api.deleteUser(userId: "user-id")
```

---

## 🎭 Работа с аватарами

### Получение метаданных аватара

```swift
let metadata = try await api.getAvatarMetadata(avatarId: "64bfa15f0e72c63d7c3934a4")
print("Gender: \(metadata.gender ?? "N/A")")
print("Body Type: \(metadata.bodyType ?? "N/A")")
print("Assets: \(metadata.assets?.count ?? 0)")
```

### Скачивание модели аватара (GLB)

```swift
// Базовое скачивание
let data = try await api.downloadAvatarModel(avatarId: "64bfa15f0e72c63d7c3934a4")

// С параметрами рендеринга
var params = AvatarRenderParams()
params.lod = 1 // Medium quality
params.pose = .aPose
params.useDracoMeshCompression = true
params.morphTargets = ["ARKit"]

let data = try await api.downloadAvatarModel(
    avatarId: "64bfa15f0e72c63d7c3934a4",
    params: params
)
```

### Скачивание модели в FBX формате

```swift
let fbxData = try await api.downloadAvatarFBX(
    avatarId: "64bfa15f0e72c63d7c3934a4",
    params: params
)
```

### Получение thumbnail (превью)

```swift
let thumbnailData = try await api.getAvatarThumbnail(
    avatarId: "64bfa15f0e72c63d7c3934a4",
    scene: .fullbodyPortraitV1,
    size: 512
)

// Конвертировать в изображение
if let image = UIImage(data: thumbnailData) {
    // Использовать изображение
}
```

### Удаление аватара (требует API ключ)

```swift
let api = ReadyPlayerMeAPI(apiKey: "your-api-key")
try await api.deleteAvatar(avatarId: "64bfa15f0e72c63d7c3934a4")
```

### Скачивание и сохранение в файл

```swift
let documentsPath = FileManager.default.urls(for: .documentDirectory, in: .userDomainMask)[0]
let fileURL = documentsPath.appendingPathComponent("avatar.glb")

try await api.downloadAndSaveAvatar(
    avatarId: "64bfa15f0e72c63d7c3934a4",
    params: params,
    to: fileURL
)
```

---

## 👤 Работа с пользователями

### Получение информации о пользователе (требует API ключ)

```swift
let api = ReadyPlayerMeAPI(apiKey: "your-api-key")
let userInfo = try await api.getUserInfo(userId: "user-id")

print("Name: \(userInfo.name ?? "N/A")")
print("Email: \(userInfo.email ?? "N/A")")
print("Avatars: \(userInfo.avatars?.count ?? 0)")
```

### Получение списка аватаров пользователя (требует API ключ)

```swift
let response = try await api.getUserAvatars(
    userId: "user-id",
    page: 1,
    limit: 20
)

print("Total avatars: \(response.pagination?.total ?? 0)")

for avatar in response.data {
    print("Avatar ID: \(avatar.id)")
    print("Created: \(avatar.createdAt ?? "N/A")")
}
```

---

## 🎨 Работа с ассетами

### Получение списка ассетов (требует API ключ)

```swift
let api = ReadyPlayerMeAPI(apiKey: "your-api-key")

// Все ассеты
let response = try await api.getAssets(page: 1, limit: 20)

// Фильтрация по типу и полу
let hairAssets = try await api.getAssets(
    type: "hair",
    gender: "male",
    page: 1,
    limit: 20
)

for asset in hairAssets.data {
    print("Name: \(asset.name)")
    print("ID: \(asset.id)")
    print("Icon: \(asset.icon ?? "N/A")")
}
```

### Получение конкретного ассета (требует API ключ)

```swift
let asset = try await api.getAsset(assetId: "asset-id")
print("Asset name: \(asset.name)")
print("Type: \(asset.type)")
```

---

## ⚙️ Параметры рендеринга

### Структура AvatarRenderParams

```swift
var params = AvatarRenderParams()

// Сцена (влияет на камеру и освещение)
params.scene = .fullbodyPortraitV1
// Варианты: .fullbodyPortraitV1, .fullbodyPortraitV2, 
//          .fullbodyPostureV1, .halfbodyPortraitV1

// Level of Detail (качество модели)
params.lod = 0 // 0 = высокое, 1 = среднее, 2 = низкое

// Поза
params.pose = .tPose // или .aPose

// Armature (тип скелета)
params.armature = .standard // или .humanoid

// Сжатие Draco (уменьшает размер файла)
params.useDracoMeshCompression = true

// Руки (добавить модель рук)
params.useHands = true

// Morph Targets для лицевой анимации
params.morphTargets = ["ARKit", "Oculus Visemes"]

// Texture Atlas (объединяет текстуры)
params.textureAtlas = .medium // .none, .high, .medium, .low

// Ограничение размера текстуры
params.textureSizeLimit = 1024

// Blend Shapes (кастомные выражения лица)
params.blendShapes = ["smile": 0.5, "eyesWide": 0.3]
```

---

## 🛠 Утилиты

### Извлечение Avatar ID из URL

```swift
let url = "https://models.readyplayer.me/64bfa15f0e72c63d7c3934a4.glb"
if let avatarId = api.extractAvatarId(from: url) {
    print("Avatar ID: \(avatarId)")
}
```

### Построение URL для Avatar Creator

Используйте `AvatarCreatorSettings` для генерации URL с конфигурацией:

```swift
var config = AvatarCreatorConfig()
config.subdomain = "companello"
config.language = .ENGLISH
config.clearCache = false
config.quickStart = false
config.gender = .MALE
config.bodyType = .FULLBODY
config.loginToken = userToken

let settings = AvatarCreatorSettings(config: config)
if let url = settings.generateUrl() {
    print("Avatar Creator URL: \(url.absoluteString)")
}
```

Или используйте существующий экземпляр `AvatarCreatorSettings`:

```swift
let settings = AvatarCreatorSettings()
// Настройте параметры через методы settings.updateLanguage() и т.д.
if let url = settings.generateUrl(token: optionalToken) {
    // Используйте URL
}
```

### Проверка наличия API ключа

```swift
if api.hasAPIKey {
    print("API key configured")
}
```

---

## 🔑 Инициализация

### Без API ключа (только публичные методы)

```swift
let api = ReadyPlayerMeAPI()

// Доступны методы:
// - Скачивание аватаров
// - Получение метаданных
// - Создание анонимных пользователей
// - Аутентификация
```

### С API ключом (все методы)

```swift
let api = ReadyPlayerMeAPI(apiKey: "your-api-key")

// Доступны все методы, включая:
// - Управление пользователями
// - Получение списка ассетов
// - Удаление аватаров
// - И другие приватные API методы
```

### С пользовательским токеном

```swift
let api = ReadyPlayerMeAPI(apiKey: userAccessToken)

// Используйте токен из login() для аутентифицированных запросов
```

---

## ⚠️ Обработка ошибок

```swift
do {
    let avatar = try await api.downloadAvatarModel(avatarId: "invalid-id")
} catch RPMAPIError.notFound {
    print("Avatar not found")
} catch RPMAPIError.unauthorized {
    print("Invalid API key or token")
} catch RPMAPIError.missingAPIKey {
    print("This operation requires an API key")
} catch RPMAPIError.networkError(let error) {
    print("Network error: \(error)")
} catch {
    print("Unknown error: \(error)")
}
```

### Типы ошибок

- `invalidURL` - Некорректный URL
- `invalidResponse` - Некорректный ответ от сервера
- `httpError(statusCode, message)` - HTTP ошибка
- `decodingError(Error)` - Ошибка декодирования JSON
- `networkError(Error)` - Сетевая ошибка
- `missingAPIKey` - Требуется API ключ
- `unauthorized` - Неверная аутентификация
- `notFound` - Ресурс не найден
- `serverError` - Ошибка сервера

---

## 📱 SwiftUI Примеры

В файле `ReadyPlayerMeAPIExample.swift` находятся:

### 1. **AvatarDownloaderView**
- Скачивание аватара по ID
- Отображение thumbnail
- Сохранение модели в файл

### 2. **AuthenticationView**
- Создание анонимного пользователя
- Email аутентификация (2-step)
- Управление токенами
- Logout

---

## 🎯 Полный пример аутентификации

```swift
// 1. Запросить код
let codeResponse = try await api.requestLoginCode(email: "user@example.com")
print("Code sent!")

// 2. Войти с кодом (получен по email)
let loginResponse = try await api.login(email: "user@example.com", code: "123456")
let accessToken = loginResponse.token
let refreshToken = loginResponse.refreshToken

// 3. Использовать access token для API запросов
let authenticatedAPI = ReadyPlayerMeAPI(apiKey: accessToken)
let userInfo = try await authenticatedAPI.getUserInfo(userId: loginResponse.user.id)

// 4. Обновить токен когда истечет
let refreshResponse = try await api.refreshToken(refreshToken: refreshToken)
let newAccessToken = refreshResponse.token
```

---

## 📝 Примечания

- **API ключ** можно получить в [Ready Player Me Studio](https://studio.readyplayer.me/)
- **Токены пользователей** имеют ограниченный срок действия - используйте refresh token
- **Anonymous users** не требуют email, но имеют ограниченный функционал
- **Avatar ID** - это 24-символьная hex строка (MongoDB ObjectID)
- Все методы используют **async/await** для асинхронных операций

---

## 🔗 Полезные ссылки

- [Ready Player Me Documentation](https://docs.readyplayer.me/)
- [REST API Reference](https://docs.readyplayer.me/ready-player-me/api-reference/rest-api)
- [Avatar Creator Configuration](https://docs.readyplayer.me/ready-player-me/integration-guides/web-and-native-integration/web-avatar-creator)

---

**Разработано для companello** 🚀

